<?php

namespace App\Http\Controllers;

use App\Models\ResellerOrderDetail;
use App\Models\ResellerOrder;
use App\Models\Product;
use Illuminate\Http\Request;

class ResellerOrderDetailController extends Controller
{
    public function index()
    {
        $details = ResellerOrderDetail::with('order.reseller', 'product')->get();

        $headers = ['Order ID', 'Reseller', 'Product', 'Qty', 'Harga', 'Subtotal', 'Aksi'];

        $rows = $details->map(function ($d) {
            $price = $d->product->reseller_price ?? $d->product->price;
            return [
                'cols' => [
                    $d->reseller_order_id,
                    $d->order->reseller->name ?? '-',
                    $d->product->name,
                    $d->qty,
                    number_format($price, 0, ',', '.'),
                    number_format($price * $d->qty, 0, ',', '.'),
                ],
                'actions' => view('components.action-button', [
                    'id' => $d->id,
                    'resource' => 'reseller-order-details'
                ])->render(),
            ];
        });

        return view('reseller_order_details.index', compact('headers', 'rows'));
    }

    public function create()
    {
        $orders = ResellerOrder::all();
        $products = Product::where('available_for_reseller', true)->get();
        return view('reseller_order_details.create', compact('orders', 'products'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'reseller_order_id' => 'required|exists:reseller_orders,id',
            'product_id' => 'required|exists:products,id',
            'qty' => 'required|integer|min:1',
        ]);

        $detail = ResellerOrderDetail::create($validated);

        $order = $detail->order;
        $order->total = $order->details->sum(function ($d) {
            $price = $d->product->reseller_price ?? $d->product->price;
            return $price * $d->qty;
        });
        $order->save();

        return redirect()->route('reseller-order-details.index')->with('success', 'Detail added successfully');
    }

    public function show(ResellerOrderDetail $resellerOrderDetail)
    {
        $resellerOrderDetail->load('order.reseller', 'product');
        return view('reseller_order_details.show', compact('resellerOrderDetail'));
    }

    public function edit(ResellerOrderDetail $resellerOrderDetail)
    {
        $orders = ResellerOrder::all();
        $products = Product::where('available_for_reseller', true)->get();
        return view('reseller_order_details.edit', compact('resellerOrderDetail', 'orders', 'products'));
    }

    public function update(Request $request, ResellerOrderDetail $resellerOrderDetail)
    {
        $validated = $request->validate([
            'reseller_order_id' => 'required|exists:reseller_orders,id',
            'product_id' => 'required|exists:products,id',
            'qty' => 'required|integer|min:1',
        ]);

        $resellerOrderDetail->update($validated);

        $order = $resellerOrderDetail->order;
        $order->total = $order->details->sum(function ($d) {
            $price = $d->product->reseller_price ?? $d->product->price;
            return $price * $d->qty;
        });
        $order->save();

        return redirect()->route('reseller-order-details.index')->with('success', 'Detail updated successfully');
    }

    public function destroy(ResellerOrderDetail $resellerOrderDetail)
    {
        $order = $resellerOrderDetail->order;
        $resellerOrderDetail->delete();

        $order->total = $order->details->sum(function ($d) {
            $price = $d->product->reseller_price ?? $d->product->price;
            return $price * $d->qty;
        });
        $order->save();

        return redirect()->route('reseller-order-details.index')->with('success', 'Detail deleted successfully');
    }
}
